
; Pool Alarm using pressure sensor


	list P=16F88
	#include p16f88.inc
	ERRORLEVEL -302
	ERRORLEVEL -306

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB0 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

;EEPROM
EEPROM1		equ	H'00'	; non-volatile storage for no pressure PWM cal ms byte
EEPROM2		equ	H'01'	; non-volatile storage for no pressure PWM cal ls byte
EEPROM3		equ	H'02'	; non-volatile storage for quiescent pressure span cal ms byte
EEPROM4		equ	H'03'	; non-volatile storage for quiescent pressure span cal ls byte
EEPROM5		equ	H'04'	; non-volatile storage for alarm pressure span cal ms byte
EEPROM6		equ	H'05'	; non-volatile storage for alarm pressure span cal ls byte

; Bank 0 RAM
TIMER		equ	H'20'	; timer switch setting
TIME_COUNT	equ	H'21'	; timer counter
TIMES_ELVN	equ	H'22'	; times 11 counter
FLASHER		equ	H'23'	; LED flasher counter
TEMP_M		equ	H'24'	; temporary ms byte
TEMP_L		equ	H'25'	; temporary ls byte
CONVERSION	equ	H'26'	; conversion timer
STORE1		equ	H'27'	; timer counter
STORE2		equ	H'28'	; timer counter
NO_PRES_M	equ	H'29'	; no pressure PWM calibration ms byte
NO_PRES_L	equ	H'2A'	; no pressure PWM calibration ls byte
QES_PRES_M	equ	H'2B'	; quiescent pressure calibration byte for monitor return function ms
QES_PRES_L	equ	H'2C'	; quiescent pressure calibration byte for monitor return function ls
ALM_PRES_M	equ	H'2D'	; alarm pressure calibration ms byte for alarm
ALM_PRES_L	equ	H'2E'	; alarm pressure calibration ls byte for alarm

HOLD		equ	H'32'	; hold or monitor
RETURN1		equ	H'33'	; return counter
RETURN2		equ	H'34'	; return tiner
QUIESC_FLS	equ	H'35'	; quiescent flash flag

SAMPLE_MIN0	equ	H'37'	; sample minimum value ms
SAMPLE_MAX0	equ H'38'	; sample maximum value ms
SPAN_MS		equ	H'39'	; current span ms
SPAN_LS		equ	H'3A'	; current span ls		
SAMP_ONCE	equ	H'3B'	; whether sampled yet flag (ie do 96 samples before comparisons made) 
INTER		equ	H'3C'	; interrupt flag
TEMPX		equ	H'3D'	; temporary register
TEMPY		equ	H'3E'	; temporary register
DEL_EXT		equ	H'3F'	; delay extension
ALARM		equ	H'40'	; alarm flag
MON_RETURN	equ	H'41'	; monitor return timer
MON_RETURN1	equ	H'42'	; monitor return timer store
TEMP_X		equ	H'43'	; temporary register
TEMP_Y		equ	H'44'	; temporary register
TEMP_Z		equ	H'45'	; temporary register
SAMPLE_MIN1	equ	H'46'	; sample minimum value ls
SAMPLE_MAX1	equ H'47'	; sample maximum value ls 
SAMPLE_MAX2	equ	H'48'	; sampled maximum value stored in interrupt ms
SAMPLE_MAX3	equ	H'49'	; sampled maximum value stored in interrupt ls
QUIESC_LEV	equ	H'4A'	; quiescent level counter
ALM_COUNT	equ	H'4B'	; alarm counter
UPDATE_QFLG equ	H'4C'	; quiescent update flag
UPDATE_AFLG equ	H'4D'	; alarm update flag	
		
; Math
SUBTRACT	equ	H'61'	; subtract value
SUBTRACT1	equ	H'62'	; subtraction value in interrupt
TEMP1		equ	H'63'	; temporary register
AARGB0		equ	H'64'	; MS Arguemnt
AARGB1		equ	H'65'	; 	
AARGB2		equ	H'66'	; 
AARGB3		equ	H'67'	; LS Argument
BARGB0		equ	H'68'	; MS Argument
BARGB1		equ	H'69'	; LS Argument
TEMPD		equ	H'6A'	; temp
TEMPB0		equ	H'6B'	; 
TEMPB1		equ	H'6C'	; 
REMB0		equ	H'6D'	; remainder
REMB1		equ	H'6E'	; 
LOOPCOUNT	equ	H'6F'	; counter 

; All Banks RAM
; Interrupt store registers 
W_TMP		equ	H'7D'	; storage of w before interrupt
STATUS_TMP	equ	H'7E'	; status storage before interrupt
FSR_STO		equ	H'7F'	; fsr storage

; Bank 2. Uses indirect addressing so there is no direct reference to the labelling

; pressure sampling. Firts the high then low byte stored
SAMP1		equ	H'110'	; sample 1
; ..............
SAMP48		equ	H'16F'	; sample 48
; Bank 3
SAMP49		equ	H'190'	; sample 49
; ...............
SAMP96		equ	H'1EF'	; sample 96

; preset EEPROM with initial settings
	ORG     H'2100'
	DE		H'6E', H'1C'; pwm
	DE		H'00', H'4A'; alarm level
	DE		H'00', H'91'; quiescent level 

; start at memory 0
	org	0				; reset vector
	goto	SETUP
	org 4				; interrupt vector 
	goto	INTERRUPT


; **********************************************************************************************
	
; lookup table for monitor return
MON_RET
	addwf	PCL,f		; add value to program counter
	retlw	D'00'		; no monitor return, setting 0
	retlw	D'3'		; 1.25 minutes, setting 1
	retlw	D'5'		; 2 minutes, setting 2
	retlw	D'8'		; 3 minutes, setting 3
	retlw	D'10'		; 4 minutes, setting 4
	retlw	D'13'		; 5 minutes, setting 5
	retlw	D'15'		; 6 minutes, setting 6
	retlw	D'18'		; 7 minutes, setting 7
	retlw	D'20'		; 8 minutes, setting 8
	retlw	D'23'		; 9 minutes, setting 9
	retlw	D'25'		; 10 minutes, setting A
	retlw	D'51'		; 20 minutes, setting B
	retlw	D'76'		; 30 minutes, setting C
	retlw	D'114'		; 45 minutes, setting D
	retlw	D'152'		; 60 minutes, setting E
	retlw	D'180'		; 75 minutes, setting F

SETUP
	clrf	PORTA		; outputs low
	bsf		STATUS,IRP	; bank 2 for indirect addressing
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'11111110'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'00111100'	; outputs (0) and inputs (1)
	movwf	TRISA
	movlw	B'00000000'	; pullups enabled
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'00011000'	; AN3, AN4 are analog inputs
	movwf	ANSEL
	movlw	B'11000000'	; right justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'11011001'	; Fosc, channel 3 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00111000'	; 0011 for 500kHz
	movwf	OSCCON		; 500kHz osc
	bcf		STATUS,RP0	; select memory bank 0

	movlw	B'00000001'	; timer 1 fosc/4
	movwf	T1CON
	bsf		STATUS,RP0	; select memory bank 1		
	movlw	H'FF'
	movwf	PR2			; PWM period register
	bcf		STATUS,RP0	; memory bank 0

; read stored EEPROM values and place in registers

	movlw	EEPROM1		; get value
	call	EEREAD
	movwf	NO_PRES_M	; no pressure PWM calibration ms byte
	movlw	EEPROM2		; 
	call	EEREAD
	movwf	NO_PRES_L	; no pressure PWM calibration ls byte

	movlw	EEPROM3		; get value
	call	EEREAD
	movwf	QES_PRES_M	; quiescent pressure span calibration byte for monitor return function ms
	movlw	EEPROM4		; get value
	call	EEREAD
	movwf	QES_PRES_L	; quiescent pressure span calibration byte for monitor return function ls

	movlw	EEPROM5		; 
	call	EEREAD
	movwf	ALM_PRES_M	; alarm pressure span calibration byte for alarm ms
	movlw	EEPROM6		; 
	call	EEREAD
	movwf	ALM_PRES_L	; alarm pressure span calibration byte for alarm ls

; pwm set
	movf	NO_PRES_M,w	; no pressure PWM calibration ms byte
	movwf	CCPR1L		; PWM
	bcf		CCP1CON,4
	bcf		CCP1CON,5	; clear 10-bits
	bsf		T2CON,2		; enable timer 2

	movlw	B'00001100'	; set PWM mode
	movwf	CCP1CON		; enable PWM operation 490Hz
	btfsc	NO_PRES_L,4	; ls byte if set then set bit 4
	bsf		CCP1CON,4
	btfsc	NO_PRES_L,5
	bsf		CCP1CON,5
	
; initialise ports
	movlw	B'00000000'	; RA0,RA1,RA6, RA7 low
	movwf	PORTA		; LEDs and alarm off
; initial settings
;	clrf	STATUS_IND	; status indication
	clrf	HOLD		; monitor when clear
	clrf	TIMES_ELVN	; alarm counter
	clrf	TIME_COUNT	; alarm counter
	clrf	RETURN1		; return timer
	clrf	RETURN2		; return counter
	clrf	SPAN_MS		; zero span
	clrf	SPAN_LS
	clrf	SAMP_ONCE
	clrf	ALARM		; alarm
	clrf	QUIESC_FLS	; quiescent flash flag
	clrf	QUIESC_LEV	; quiescent level counter
	clrf	ALM_COUNT	; alarm counter
	clrf	UPDATE_QFLG	; quiescent update flag
	clrf	UPDATE_AFLG	; alarm update flag
	movlw	H'10'		; start of indirect address H110
	movwf	FSR

; start up delay
POWER_ON
	movlw	D'25'		; approx 120ms 
	call	DELAYX		; delay

; test quiescent switch. If pressed then adjust pwm so AN3 is at 2V
	btfsc	PORTB,2		; when pressed adjust offset
	goto	POWER_UP_DELAY

; OFFSET ADJUSTMENT		; sets offset so AN3 is about 2.2V with no pressure at sensor 
	bsf		PORTA,0		; set status LED
	clrf	CCPR1L		; clear PWM output
	bcf		CCP1CON,4
	bcf		CCP1CON,5
; bit 7
	bsf		CCPR1L,7	; start at ms bit high (50% PWM)
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,7	; clear bit 7 
; bit 6
	bsf		CCPR1L,6	; set bit 6, bit 7 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,6	; clear bit 6 
; bit 5
	bsf		CCPR1L,5	; set bit 5, bit 6 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,5	; clear bit 5 
; bit 4
	bsf		CCPR1L,4	; set bit 4, bit 5 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,4	; clear bit 4 
; bit 3
	bsf		CCPR1L,3	; set bit 3, bit 4 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,3	; clear bit 3 
; bit 2
	bsf		CCPR1L,2	; set bit 2 bit 3 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,2	; clear bit 2 
; bit 1
	bsf		CCPR1L,1	; set bit 1, bit 2 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,1	; clear bit 1 
; bit 0
	bsf		CCPR1L,0	; set bit 0, bit 1 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCPR1L,0	; clear bit 0 
; bit 9
	bsf		CCP1CON,5	; set bit 5 (9th bit), bit 0 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCP1CON,5	; clear bit 5 
; bit 10
	bsf		CCP1CON,4	; set bit 4 (10th bit), bit 5 set or clear depending on comparison with required value	
	call	READ_COMP	; read AN3 and compare value
; compare value
	btfsc	SUBTRACT,7	; if negative 
	bcf		CCP1CON,4	; clear bit 4 
; PWM adjustment complete
; store values
	movf	CCPR1L,w	; PWM
	movwf	NO_PRES_M	; store
	movf	CCP1CON,w	; ls bits
	movwf	NO_PRES_L	; store
	movlw	EEPROM1		; get value
	call	EEREAD		; sets EEADR
	movf	NO_PRES_M,w	; no pressure PWM calibration ms byte
	call	EWRITE		; write to EEPROM
	movlw	EEPROM2		; 
	call	EEREAD		; sets EEADR
	movf	NO_PRES_L,w	; no pressure PWM calibration ls byte
	call	EWRITE

WAIT_QES
	movlw	D'50'
	call	DELAYX		; delay to debounce
	btfss	PORTB,2		; wait till quiescent switch is open
	goto	WAIT_QES
	bcf		PORTA,0		; status LED off
	goto	SET_INTERRUPTS

; subroutine to read values
READ_COMP
	movlw	D'08'
	movwf	DEL_EXT		; delay extension for pulse width filter to settle
DEL_X	
	movlw	D'255'		;  
	call	DELAYX		; delay
	decfsz	DEL_EXT,f
	goto	DEL_X		; continue delay	
	call	CH_3AD		; measure offset TEMP_M
; compare with 2.5V
	movf	TEMP_M,w	; current pressure ms byte
	sublw	H'2'		; ms bit of D512 (2.5V) 
	movwf	SUBTRACT	; subtract register
	movf	TEMP_L,w	; current pressure ls byte
	sublw	H'00'		; 2.5V = H200
	btfss	STATUS,C	; decrement subtract register if over
	decf	SUBTRACT,f	; if bit 7 set then pressure >2.5V
	return

POWER_UP_DELAY
	bsf		PORTA,1		; Hold LED on (indicates power at startup)
	movlw	D'10'
	movwf	DEL_EXT		; delay extension for pulse width filter to settle
DEL_Y	
	movlw	D'255'		; approx 10s  
	call	DELAYX		; delay
	decfsz	DEL_EXT,f
	goto	DEL_Y		; continue delay	
	bcf		PORTA,1		; LED off

; allow interrupts
SET_INTERRUPTS
	bsf		STATUS,RP0		; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 overflow interrupt
	bcf		STATUS,RP0		; select memory bank 0
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag
	bsf		INTCON,PEIE		; enable periperal interrupts 	 
	bsf		INTCON,GIE

START_SAMP_CHECK; check when all samples have been made
	btfsc	SAMP_ONCE,0		; when samples have been made can start
	goto	START
	bcf		PORTA,6			; alarm off
	goto	START_SAMP_CHECK

START
; test hold/monitor switch
	btfss	PORTB,3			
	goto	HOLD_MON		; HOLD/MONITOR switch
; test quiescent switch
	btfss	PORTB,2
	goto	QUIESC_MON		; quiescent switch
; test calibrate switch
	btfss	PORTB,1
	goto	CAL_MON			; calibrate switch

; compare current pressure with calibration pressure value. If over then alarm
	movf	SPAN_MS,w		; current pressure span ms byte
	subwf	ALM_PRES_M,w	; calibration pressure for alarm ms
	movwf	SUBTRACT		; subtract register
	movf	SPAN_LS,w		; current pressure span ls byte
	subwf	ALM_PRES_L,w	; calibration pressure for alarm ls
	btfss	STATUS,C		; decrement subtract register if over
	decf	SUBTRACT,f
	btfss	SUBTRACT,7		; if set then alarm	
 	goto	ALM_BY
; alarm
; if already an alarm, bypass alarm counter update
	btfsc	ALARM,0
	goto	ALM_BY
	btfss	UPDATE_AFLG,0	; alarm update flag if set then increase count
	goto	BY_INC_A_COUNT
	incf	ALM_COUNT,f		; increase alarm counter
	clrf	UPDATE_AFLG		; clear ready for set in interrupt
BY_INC_A_COUNT 
	movf	ALM_COUNT,w
	sublw	D'03'			; if more than 3 times over then set alarm
	btfss	STATUS,C
	goto	ALM_BY1		
	call	CH_4AD			; read VR1 and set in TIME_COUNT
	clrf	TIMES_ELVN		; zero counters for alarm period set	
; check for monitor or hold
	btfss	HOLD,0			; when bit is clear it is monitor
	bsf		PORTA,6			; alarm siren when in monitor (not if hold)
	bsf		ALARM,0			; set alarm flag
	goto	ALM_BY1

ALM_BY
	clrf	ALM_COUNT		; alarm counter
ALM_BY1
; check for monitor or hold
	btfss	HOLD,0			; when bit is clear it is monitor
	goto	STILL_ALM		; 
; hold
; if an alarm condition and levels have dropped switch alarm off
	movf	SPAN_MS,w		; current pressure span ms byte
	subwf	ALM_PRES_M,w	; alarm pressure for quiescent ms
	movwf	SUBTRACT		; subtract register
	movf	SPAN_LS,w		; current pressure span ls byte
	subwf	ALM_PRES_L,w	; alarm pressure for quiescent ls
	btfss	STATUS,C		; decrement subtract register if over
	decf	SUBTRACT,f
	btfsc	SUBTRACT,7		; if set then alarm
	goto	STILL_ALM		; still an alarm condition
	clrf	ALARM			; alarm cleared	
	bcf		PORTA,7			; alarm LED off
	bcf		PORTA,6			; ensure alarm is off
	clrf	TIME_COUNT		; alarm time counter
STILL_ALM
; monitor span during hold to top up RETURN2 with monitor return value each time pool movement 
; is above quiescent (get switch position then call MON_RET lookup table place in RETURN2)
	movf	SPAN_MS,w		; current pressure span ms byte
	subwf	QES_PRES_M,w	; calibration pressure for quiescent ms
	movwf	SUBTRACT		; subtract register
	movf	SPAN_LS,w		; current pressure span ls byte
	subwf	QES_PRES_L,w	; calibration pressure for quiescent ls
	btfss	STATUS,C		; decrement subtract register if over
	decf	SUBTRACT,f
	btfss	SUBTRACT,7		; if set then movement
 	goto	NO_MOVEMENT
; pool movement
	call	READ_MON_RETURN_SWITCH; read switch 
	movf	MON_RETURN1,w	; monitor return switch setting
	call	MON_RET			; get timeout value
	movwf	RETURN2			; timeout
	clrf	RETURN1
	bsf		QUIESC_FLS,0	; set quiescent flag indicating above quiescent
	btfss	UPDATE_QFLG,0	; if quiescent update flag set increase counter
	goto	BY_INC_Q
	incf	QUIESC_LEV,f	; increase when there is above quiescent pool movement
	clrf	UPDATE_QFLG
BY_INC_Q	
	goto	NO_MOVEMENT1
NO_MOVEMENT
	clrf	QUIESC_LEV		; clear quiescent counter
	clrf	QUIESC_FLS		; quiescent flash flag
NO_MOVEMENT1

; auto return to Hold when quiescent levels are high
; If monitor mode and PORTA,2 is clear and the quiescent levels are high switch to hold 
; (but not if during alarm period)
MONIT_OFF
	btfsc	ALARM,0			; if an alarm, then bypass
	goto	START
	btfsc	HOLD,0			; if set then hold		
	goto	START			; not monitor mode
; monitor mode
	btfsc	PORTA,2			; if set then no return to hold 
	goto	START

	movf	QUIESC_LEV,w	; if over 15-consecutive 2-second samples that are
; above quiescent in level then set to hold 
	sublw	D'15'
	btfsc	STATUS,C
	goto 	START

; if over then set to Hold. This is done in "HOLD_MON" below

; switch actions
HOLD_MON
; If Hold/Monitor switch, clear alarm and timers (Time_count), (For monitor to hold set RETURN2 value)
; toggle from Hold to Monitor or monitor to hold.
	bcf		PORTA,6		; alarm off
	bcf		PORTA,7		; alarm LED off
	bcf		STATUS,GIE	; stop interrupt
	clrf	ALARM
	clrf	TIME_COUNT	; alarm time counter
	clrf	SPAN_MS		; span to 0
	clrf	SPAN_LS
	movlw	H'10'
	movwf	FSR			; reset FSR for samples
	bsf		STATUS,GIE	; enable interrupt

; toggle HOLD
	movf	HOLD,w		; clear is monitor, bit 0 set is hold
	movlw	D'00'		; clear HOLD
	movwf	HOLD		; clear HOLD remains clear if was not 0 
	movlw	D'01'		; ready to set to 1
	btfsc	STATUS,Z	
	movwf	HOLD		; if was clear then set to 1 
	
; if hold then set monitor return timer
	btfss	HOLD,0		; hold if bit 0 set
	goto	SW_HOLD_MON
	call	READ_MON_RETURN_SWITCH; read switch 
	movf	MON_RETURN1,w; monitor return switch setting
	call	MON_RET		; get timeout value
	movwf	RETURN2		; timeout
	clrf	RETURN1

SW_HOLD_MON
	clrf	QUIESC_FLS	; quiescent flag cleared
; wait till switch is open

	movlw	D'50'
	call	DELAYX		; delay to debounce
	btfss	PORTB,3			
	goto	SW_HOLD_MON	
	goto	START

QUIESC_MON
; If quiescent switch closed then monitor AN3 for min and max readings. 
; Calculate span and multiply by 1.05 (for headroom). Set status LED
	bcf		STATUS,GIE	; stop interrupt
	movlw	H'10'
	movwf	FSR			; set at start of sample registers
	clrf	SAMP_ONCE	; clear sample flag so samples are started afresh
	bsf		STATUS,GIE	; allow interrupts

RECH_SAMP0
	bsf		PORTA,0		; status LED on
	btfss	SAMP_ONCE,0	; wait till set when all samples made and span is calculated (in interrupt)
	goto	RECH_SAMP0
	bcf		STATUS,GIE	; stop interrupt
; multiply by 105%
	movf	SPAN_MS,w	; get ms span value
	movwf	AARGB0
	movf	SPAN_LS,w	; get ls span value
	movwf	AARGB1
	bsf		STATUS,GIE	; allow interrupts
	movlw	D'105'		; x 105
	movwf	BARGB1
	clrf	BARGB0
	call 	FXM1616U	; multiply
; shift values
	movf	AARGB1,w
	movwf	AARGB0		; shift 1-0
	movf	AARGB2,w
	movwf	AARGB1		; 2-1
	movf	AARGB3,w
	movwf	AARGB2		; 3-2	
	movlw	D'100'		; divisor
	movwf	BARGB1
	clrf	BARGB0
	call	FXD2416U	; divide
	movf	AARGB1,w	; 
	movwf	QES_PRES_M
	movf	AARGB2,w
	movwf	QES_PRES_L
; store
STO_QUIES
	movlw	EEPROM3		; get value
	call	EEREAD
	movf	QES_PRES_M,w; quiescent pressure span calibration byte for monitor return function
	call	EEWRITE	
	movlw	EEPROM4		; get value
	call	EEREAD
	movf	QES_PRES_L,w; quiescent pressure span calibration byte for monitor return function
	call	EEWRITE	
	
	bcf		PORTA,0		; status LED off

; wait till quiescent switch is open
SW_QUIESC_MON
	movlw	D'50'
	call	DELAYX		; delay to debounce
	btfss	PORTB,2
	goto	SW_QUIESC_MON
	goto	START

CAL_MON
; If Calibrate pressed monitor AN3 for min amd max readings. Calculate span and mult by 0.95 for headroom 

	bcf		STATUS,GIE	; stop interrupt
	movlw	H'10'
	movwf	FSR			; set at start of sample registers
	clrf	SAMP_ONCE	; clear sample flag so samples are started afresh
	bsf		STATUS,GIE	; allow interrupts
RECH_SAMP1
	bsf		PORTA,0		; status LED on
	btfss	SAMP_ONCE,0	; wait till set when all samples made and span is calculated (in interrupt)
	goto	RECH_SAMP1
	bcf		STATUS,GIE	; stop interrupt
; multiply by 95%
	movf	SPAN_MS,w	; get ms span value
	movwf	AARGB0
	movf	SPAN_LS,w	; get ls span value
	movwf	AARGB1
	bsf		STATUS,GIE	; allow interrupts
	movlw	D'95'		; x 95
	movwf	BARGB1
	clrf	BARGB0
	call 	FXM1616U	; multiply
; shift values
	movf	AARGB1,w
	movwf	AARGB0		; shift 1-0
	movf	AARGB2,w
	movwf	AARGB1		; 2-1
	movf	AARGB3,w
	movwf	AARGB2		; 3-2	
	movlw	D'100'		; divisor
	movwf	BARGB1
	clrf	BARGB0
	call	FXD2416U	; divide
	movf	AARGB1,w	; 
	movwf	ALM_PRES_M
	movf	AARGB2,w
	movwf	ALM_PRES_L
; store
STO_ALM
	movlw	EEPROM5		; get value
	call	EEREAD
	movf	ALM_PRES_M,w; alarm pressure span calibration byte for monitor return function
	call	EEWRITE	
	movlw	EEPROM6		; get value
	call	EEREAD
	movf	ALM_PRES_L,w; alarm pressure span calibration byte for monitor return function
	call	EEWRITE	
	
	bcf		PORTA,0		; status LED off

; wait till calibrate switch is open
SW_CAL_MON
	movlw	D'50'
	call	DELAYX		; delay to debounce
	btfss	PORTB,1
	goto	SW_CAL_MON	; calibrate switch
	goto	START

; **********************************************************************
; INTERRUPT

; start interrupt by saving w and status registers 
	
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP		; status in status_tmp 
	bcf		STATUS,RP0		; bank select
	bcf 	STATUS,RP1		; select memory bank 0 
	bsf		STATUS,IRP		; indirect addressing 
	btfss	PIR1,TMR1IF		; timer 1 interrupt flag
	goto	CHECK_END		; no timer interrupt so exit
	bcf		PIR1,TMR1IF		; clear flag
	bsf		INTER,0			; set interrupt flag

; on overflow set timer to 12265 so 4x 1/500kHz = 8us. 8us x 12265 = every .09812s 
; a/d value (0-5V) x 12 x .09812 = timeout
; 5V or 255 x 12 x .09812 = 300s or 5 minutes. So 1V = 1 minute
 
	bcf		T1CON,TMR1ON	; stop timer
	movlw	H'D0'
	movwf	TMR1H			; timer high byte
	movlw	H'16'			; 
	movwf	TMR1L			; timer low byte
	bsf		T1CON,TMR1ON	; restart timer	
	
; sample pressure and store in a 96 16-byte long memory store H110 to H16F and H190 to H1EF (bank 2)
	call	CH_3AD			; value in TEMP_M, TEMP_L
	movf	TEMP_M,w		; ms A/D value
	movwf	INDF			; place in storage
	incf	FSR,f			; next location
	movf	TEMP_L,w		; ls A/D
	movwf	INDF			; store
	incf	FSR,f			; next location
; check sample flag
	btfss	SAMP_ONCE,0		; if set sample over 20 16-bit locations
	goto	SAMPX
	movf	FSR,w
	sublw	H'38'			; if at H139 set at H10
	movlw	H'10'			; ready to store in FSR if needed
	btfss	STATUS,C
	movwf	FSR				; 
	goto	END_SAMP 
SAMPX 
; check which memory bank
	btfsc	FSR,7			; if bit 7 is set then bank 4
	goto	BNK_4	
; check for end of bank 3 at H16F. If so set at start of bank 3 H190
	movf	FSR,w
	sublw	H'6F'			; if at H170 set at H190
	movlw	H'90'			; ready to store in FSR if needed
	btfss	STATUS,C
	movwf	FSR				; set at H90 the H190 set by IRP
	goto	END_SAMP 
; check for end of bank 4 at H1EF. If so return to H110
BNK_4
	movf	FSR,w
	sublw	H'EF'			; if at HF0 set at 10
	movlw	H'10'			; ready to store in FSR if needed
	btfss	STATUS,C
	movwf	FSR				; return to H10 

; when returned to H110 indicate all samples have started with samp_once flag
END_SAMP
	movlw	H'10'
	xorwf	FSR,w			; check sampled once flag when returned to H10
	btfss	STATUS,Z	
	goto	FIND_MIN_MAX
	bsf		SAMP_ONCE,0		; set flag when all samples have been made at least once
	bsf		UPDATE_QFLG,0	; quiescent update flag

; ie flag is clear until 96 samples made
FIND_MIN_MAX
; find minimum and maximum from samples	when sample flag set
	btfss	SAMP_ONCE,0		; when set do comparisons
	goto	END_MIN_MAX

; initialise FSR
	movf	FSR,w			; current fsr
	movwf	FSR_STO			; store FSR to return to later
	movlw	H'10'			; start at H110
	movwf	FSR				; starting address

; initialise sample min and max values
	clrf	SAMPLE_MAX0		; clear max value to minimum ms byte
	clrf	SAMPLE_MAX1		; ls byte
	movlw	B'00000011'		; set minimum value to max
	movwf	SAMPLE_MIN0
	movlw	H'FF'			; ls bits set
	movwf	SAMPLE_MIN1

; start comparisons
NXT_COMPARISON
; get store value
	movf	INDF,w			; value
	movwf	TEMPX			; ms store
	incf	FSR,f			; next location
	movf	INDF,w			; next value ls byte
	movwf	TEMPY			; ls store
; compare
	movf	TEMPX,w			; sampled pressure span ms byte
	subwf	SAMPLE_MAX0,w	; max ms pressure
	movwf	SUBTRACT1		; subtract register
	movf	TEMPY,w			; current pressure span ls byte
	subwf	SAMPLE_MAX1,w	; max ls pressure
	btfss	STATUS,C		; decrement subtract register if over
	decf	SUBTRACT1,f
	btfss	SUBTRACT1,7		; if set then greater	
 	goto	NOT_GREATEST
	movf	TEMPX,w			; place greater value in greater register
	movwf	SAMPLE_MAX0		; max value
	movf	TEMPY,w
	movwf	SAMPLE_MAX1
	goto	NXT_VALUE

; check min value
NOT_GREATEST
	movf	TEMPX,w			; sampled pressure span ms byte
	subwf	SAMPLE_MIN0,w	; min ms pressure
	movwf	SUBTRACT1		; subtract register
	movf	TEMPY,w			; sampled pressure span ls byte
	subwf	SAMPLE_MIN1,w	; min ls pressure
	btfss	STATUS,C		; decrement subtract register if over
	decf	SUBTRACT1,f
	btfsc	SUBTRACT1,7		; if set then greater	
	goto	NXT_VALUE
	movf	TEMPX,w			; place smaller value in smaller register
	movwf	SAMPLE_MIN0		; min value
	movf	TEMPY,w
	movwf	SAMPLE_MIN1

NXT_VALUE
	incf	FSR,f			; next value
; check sample flag
	btfss	SAMP_ONCE,0		; if set sample over 20 16-bit locations
	goto	SAMPY
	movf	FSR,w
	sublw	H'38'			; if at H139 set to original FSR
	btfsc	STATUS,C
	goto	NXT_COMPARISON
	goto	RETURN_FSR 
SAMPY 	
; check which bank
	btfsc	FSR,7			; if bit 7 is set then bank 4
	goto	BANK_4	
; check for end of bank 3 at H16F. If so set at start of bank 3 H190
	movf	FSR,w
	sublw	H'6F'			; if at H70 set at H90
	movlw	H'90'			; ready to store in FSR if needed
	btfss	STATUS,C
	movwf	FSR				; set at H90
	goto	NXT_COMPARISON 
; check for end of bank 4 at H1EF. If so return to H110
BANK_4
	movf	FSR,w
	sublw	H'EF'			; if at HF0 set to original FSR
	btfsc	STATUS,C
	goto	NXT_COMPARISON
RETURN_FSR 
; return FSR
	movf	FSR_STO,w
	movwf	FSR				; restore FSR
	
; calculate span. Take min from max
	btfss	SAMP_ONCE,0		; when set calc span
	goto	END_MIN_MAX
	movf	SAMPLE_MIN0,w	; sampled min ms byte
	subwf	SAMPLE_MAX0,w	; sampled max ms pressure
	movwf	SUBTRACT1		; subtract register
	movf	SAMPLE_MIN1,w	; sampled min ls byte
	subwf	SAMPLE_MAX1,w	; sampled max ls pressure
	btfss	STATUS,C		; decrement subtract register if over
	decf	SUBTRACT1,f
 	movwf	SPAN_LS			; w had subtract value ls byte
	movf	SUBTRACT1,w
	movwf	SPAN_MS			; current pressure span	
	movf	SAMPLE_MAX0,w	; sampled max value
	movwf	SAMPLE_MAX2		; stored for use outside interrupt
	movf	SAMPLE_MAX1,w	; sampled max value
	movwf	SAMPLE_MAX3		; stored for use outside interrupt
	bsf		UPDATE_AFLG,0	; alarm update flag

END_MIN_MAX
; timers	

; flasher timer
	incf	FLASHER,f		; flasher timer
; x11 counter	
	incf	TIMES_ELVN,f	; counts to 12
	movf	TIMES_ELVN,w	; to w for comparison
	sublw	D'11'			; 
	btfss	STATUS,C
	clrf	TIMES_ELVN		; cleared when reaches 12

	movf	TIMES_ELVN,w
	btfss	STATUS,Z		; when zero change TIME_COUNT
	goto	BY_ALM_OFF
; alarm time counter
	movf	TIME_COUNT,w
	btfss	STATUS,Z		; when zero do not decrement
	decf	TIME_COUNT,f	; timeout
	
	movf	TIME_COUNT,w
	btfss	STATUS,Z		; when zero clear alarm
	goto	BY_ALM_OFF

; if changing from alarm on to alarm off (ie at end of alarm period)
; and if return to hold function enabled,  return to hold

	btfss	PORTA,6			; if alarm is on
	goto	QUIET_ALM
	btfss	PORTA,2			; if set then no return to hold 
; return to hold
	bsf		HOLD,0			; set to hold

QUIET_ALM	
	clrf	ALARM			; alarm flag
	bcf		PORTA,6			; alarm off
	bcf		PORTA,7			; alarm LED off

BY_ALM_OFF
; monitor return timer
; if Hold then monitor return timer is topped up while ever pool movement is above quiescent
; decrease return timers til zero then set to monitor  	
	
	decfsz	RETURN1,f		; reduce to 0 (0-255 counter)
	goto	NO_DEC2
	movf	RETURN2,w		; return counter
	btfsc	STATUS,Z		; if zero
	goto	ZRO_COUNT
	decf	RETURN2,f
	goto	NO_DEC2
ZRO_COUNT
; check if MON_RETURN1 is zero if so then no monitor return
	movf	MON_RETURN1,w
	btfss	STATUS,Z	
	clrf	HOLD			; return to monitoring
NO_DEC2
; if alarm flash LED
		
	btfss	ALARM,0
	goto	STAT_LED
	clrf	QUIESC_FLS		; quiescent indicator off
	btfss	FLASHER,0		; if flasher counter bit 0 set set LED
	goto	CLR_ALM_LED
	bsf		PORTA,7			; alarm LED on
	goto	STAT_LED
CLR_ALM_LED
	bcf		PORTA,7			; alarm LED off	

STAT_LED
; allow quiescent flash if QUIESC_FLS,0 is set
	btfss	QUIESC_FLS,0	; quiescent flag
	goto	STAT_LED_OFF
	btfss	FLASHER,0		; if flasher counter bit 0 set set LED
	goto	CLR_QUI_LED
	bsf		PORTA,0			; quiesc LED on
	goto	HOLD_CK
CLR_QUI_LED
	bcf		PORTA,0			; quiesc LED off	
	goto	HOLD_CK

STAT_LED_OFF
	bcf		PORTA,0			; status LED off	

; check hold or monitor
HOLD_CK
	movf	HOLD,w			; if zero then hold
	btfsc	STATUS,Z		; 
	goto	MON_FLASH		; monitor is flash
	bsf		PORTA,1			; hold is LED continuously on
	goto	CHECK_END
MON_FLASH
	movf	FLASHER,w
	andlw	B'00001111'		; ms 4-bits out
	btfss	STATUS,Z		; if zero then on
	goto	OFF_HOLD
	bsf		PORTA,1
	goto	CHECK_END
OFF_HOLD
	bcf		PORTA,1

CHECK_END
; end of interrupt reclaim w and status 

	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt


; ***************************************

; Channel A/D value 
CH_3AD ; pressure input
	bcf		ADCON0,5
	bsf		ADCON0,3
	bsf		ADCON0,4	; channel 3
	call	ACQUIRE_AD	; convert
	movf	ADRESH,w
	movwf	TEMP_M
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
	movwf	TEMP_L
	return

CH_4AD ; alarm period
	bcf		INTCON,GIE	; stop interrupts because CH_3AD is run in interrupt
	nop
	bsf		ADCON0,5
	bcf		ADCON0,3
	bcf		ADCON0,4	; channel 4
	call	ACQUIRE_AD	; convert
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
	movwf	TEMP_Z		; ls byte store
	rrf		ADRESH,f	; shift ms bit right into ls byte
	rrf		TEMP_Z,f
	rrf		ADRESH,f	; shift ms bit right into ls byte
	rrf		TEMP_Z,w
	movwf	TIME_COUNT	; alarm period
	bsf		INTCON,GIE
	return

; subroutine to wait for conversion

ACQUIRE_AD
; wait for >20us
	movlw	D'5'
	movwf	CONVERSION
LOOP_CONV
	decfsz	CONVERSION,f	; decrease 
	goto	LOOP_CONV	
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return

; delay loop 

DELAY
	movlw	D'4'		; delay value
DELAYX
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'165'
DELDSP
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8		; decrease till STORE1 is zero
	return

; subroutine to read EEPROM memory 

EEREAD
	bcf		STATUS,RP0	; select bank 0
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE; (Without interrupt)
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR2	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR2			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit

WRITE1
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE1		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	return				; value written 

EEWRITE	;(with interrupt)
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit

WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	bsf		INTCON,GIE	; allow interrupts	
	return				; value written 

; read monitor return timer switch
READ_MON_RETURN_SWITCH
	clrf	MON_RETURN	; monitor return timer switch
	btfss	PORTB,4		; if clear set bit 0	
	bsf		MON_RETURN,0
	btfss	PORTB,6		; if clear set bit 1	
	bsf		MON_RETURN,1
	btfss	PORTB,7		; if clear set bit 2	
	bsf		MON_RETURN,2
	btfss	PORTB,5		; if clear set bit 3	
	bsf		MON_RETURN,3
	movf	MON_RETURN,w
	movwf	MON_RETURN1	; store value that does not alter during setting procedure
	return

         
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   H'00'
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN


; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U        CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LA 		RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L8         RLF             AARGB1, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LB         RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L16        RLF             AARGB2, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LC 		RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        	ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L			RETURN
	

 end
